// Copyright 2014 Google Inc. All Rights Reserved.

#include "WorkQueue.h"
#include "Log.h"

#define MAX_WQ_THREAD_NAME_SIZE     32

void WorkQueueThread::run() {
    while (!mQueue->mIsStopping) {
        shared_ptr<WorkItem> item = mQueue->dequeueWork();
        if (item == NULL || mQueue->mIsStopping) {
            // Happens while shutting down.
            break;
        }
        item->run();
        // in case of wait shutdown (dequeue all work items)
        // go quickly through the queue without yield (incoming signal)
        if (!mQueue->mIsWaitStopping) {
            // only yield if running. not while shutting down.
            yield();
        }
    }
}

WorkQueue::~WorkQueue() {
    if (mThreads != NULL) {
        LOGW("Workqueue went out of scope without calling shutdown!");
    }
}

bool WorkQueue::start() {
    mThreads = new WorkQueueThread[mNumThreads];
    bool ret = true;
    for (int i = 0; i < mNumThreads; i++) {
        // Slightly idiotic that c++ has no good way to call the parameterized
        // constructor while allocating an array of objects.
        mThreads[i].setQueue(this);
        ret &= mThreads[i].start();
    }
    return ret;
}

void WorkQueue::queueWork(const shared_ptr<WorkItem>& work, unsigned int workItemDataLen) {
    int workItemSize = 0;
    bool wasUnderThreshold = false;
    {
        // Keep the sem-up outside the scope of mLock to prevent unnecessary lock nesting.
        Autolock l(&mLock);
        mWorkItems.push_back(work);

        if (mBytesBeforeThreshold < mThresholdValue) {
            mBytesBeforeThreshold += workItemDataLen;
            wasUnderThreshold = true;
        }
        workItemSize = (int)mWorkItems.size();
    }

    // Keep the sem-up even if no threshold was defined (default set to 0)
    if (mBytesBeforeThreshold >= mThresholdValue) {
        if (wasUnderThreshold) {
            LOGD("queueWork()  workqueue %p buffering completed. mThresholdValue =%d, currValue=%d (items=%d).",
                    this, mThresholdValue, mBytesBeforeThreshold, workItemSize);

            // sem-up all items before threshold
            for (int i = 0; i < workItemSize; i++)
                mAvailable.up();
        }
        else {
            // if we weren't before the threshold then just do only one sem-up
            mAvailable.up();
        }
    }
}

shared_ptr<WorkItem> WorkQueue::dequeueWork() {
    if (!mIsWaitStopping) {
        mAvailable.down();
    } else {
        LOGD("dequeueWork()  workQueue %p mIsWaitStopping was set, ignore sem_wait", this);
    }
    Autolock l(&mLock);
    if (mWorkItems.size() == 0) {
        LOGD("dequeueWork()  workQueue %p is empty!", this);
        return NULL;
    }
    shared_ptr<WorkItem> work = mWorkItems.front();
    mWorkItems.pop_front();
    return work;
}

void WorkQueue::waitShutdown() {
    mIsWaitStopping = true;
    for (int i = 0; i < mNumThreads; i++) {
        {
            Autolock l(&mLock);
            if (mWorkItems.size() > 0)
                LOGD("waitShutdown()  workItems = %d !", mWorkItems.size());
        }
        mAvailable.up();
    }
    for (int i = 0; i < mNumThreads; i++) {
        mThreads[i].join();
    }
    delete[] mThreads;
    mThreads = NULL;
    mBytesBeforeThreshold = 0;
}

void WorkQueue::shutdown() {
    mIsStopping = true;
    {
       Autolock l(&mLock);
       if (mWorkItems.size() > 0)
           LOGD("Shutdown()  workItems = %d !", mWorkItems.size());
    }
    for (int i = 0; i < mNumThreads; i++) {
        mAvailable.up();
    }
    for (int i = 0; i < mNumThreads; i++) {
        mThreads[i].join();
    }
    delete[] mThreads;
    mThreads = NULL;
    mBytesBeforeThreshold = 0;
}

void WorkQueue::setName(const char* name) {
    char str[MAX_WQ_THREAD_NAME_SIZE];
    for (int i = 0; i < mNumThreads; i++) {
        snprintf(str, sizeof(str), "%s-%d", name, i);
        mThreads[i].setName(str);
    }
}

uint32_t WorkQueue::getNumOfWorkItems()
{
    Autolock l(&mLock);
    if (!mWorkItems.empty())
    {
        return (uint32_t)mWorkItems.size();
    }
    else
    {
        return 0;
    }
}
